#include "gui.h"
#include "atari_fn.h"
#include <atari.h>
#include <joystick.h>
#include <peekpoke.h>
#include <time.h>
#include <string.h>

//////////////////////////////////////////////////////////////////////////

#define SAVMSC *(unsigned int *) 88 // screen address
byte *video_ptr; 
extern char joy_driver;
unsigned char joy;

//////////////////////////////////////////////////////////////////////////

extern byte local_x, local_y, local_temp1;
extern byte visible_board[8][8];
extern byte visible_board_white_pieces;
extern byte visible_board_black_pieces;
extern byte move_x;
extern byte move_y;
extern byte last_x;
extern byte last_y;
extern byte last_color;
extern byte current_color;
extern byte game_turn;

extern byte current_color;
extern bool black_computer_controlled;
extern bool white_computer_controlled;
extern byte tree_depth;

extern void init_first_board();
extern void copy_new_board_to_visible();
extern byte find_possible_moves_and_count_pieces();

#define SetChar(x,y,a) video_ptr[(x)+(y)*40]=(a);
#define GetChar(x,y) video_ptr[(x)+(y)*40]

byte my_random(byte range)
{
	if (range==0)
		return 0;
	else
		return ((PEEK(0xD20A))%range);
}

void wait(unsigned char t)
{	
	POKE(CDTMV3,t*5);
	while(PEEK(CDTMV3))
		joy=joy_read(JOY_1);
}


void print_value(unsigned char x,unsigned char y,unsigned char val)
{
	if (val>9)
		SetChar(x++,y,val/10+0x30-32);
	SetChar(x,y,val%10+0x30-32);
}

void wait_for_fire()
{
	do 
	{
		wait(1);
	} while (!JOY_BTN_FIRE(joy));
	do 
	{
		wait(1);
	} while (JOY_BTN_FIRE(joy));
}

void print_string(unsigned char x,unsigned char y,char *s, bool inverse)
{
	do
	{
		if (*s < 32)
		{
			SetChar(x++,y,*s+64+(inverse*128));
		}
		else if (*s<96)
		{
			SetChar(x++,y,*s-32+(inverse*128));
		}
		else
		{
			SetChar(x++,y,*s+(inverse*128));
		}
		++s;
	} while(*s!=0);
}


void show_board()
{
	for (local_y=0;local_y<8;++local_y)
	{
		for (local_x=0;local_x<8;++local_x)
		{
			switch(visible_board[local_y][local_x])
			{
			case WHITE:
				local_temp1=68;
				break;
			case BLACK:
				local_temp1=72;
				break;
			case EMPTY:
				local_temp1=64;
				break;
			case POSSIBLE_MOVE:
				local_temp1=84;
				break;
			}
			SetChar(1+local_x*2,1+local_y*2,local_temp1);
			SetChar(2+local_x*2,1+local_y*2,local_temp1+1);
			SetChar(1+local_x*2,2+local_y*2,local_temp1+2);
			SetChar(2+local_x*2,2+local_y*2,local_temp1+3);
		}
	}	
}

void show_score()
{
	print_string(20,5,"WHITE PIECES:   ",false);
	print_string(20,7,"BLACK PIECES:   ",false);

	print_value(34,5,visible_board_white_pieces);
	print_value(34,7,visible_board_black_pieces);

	if (current_color==WHITE)
		print_string(20,10,"WHITE TURN",false);
	else
		print_string(20,10,"BLACK TURN",false);

	print_value(31,10,game_turn);
}

void draw_game_screen() // only once
{
	// draw board border

	for (local_x=1;local_x<8*2+1;++local_x)
	{
		SetChar(local_x,0,94);
		SetChar(local_x,8*2+1,95);

		SetChar(0,local_x,88);
		SetChar(8*2+1,local_x,89);
	}
	SetChar(0,0,90);
	SetChar(0,8*2+1,92);

	SetChar(8*2+1,0,91);
	SetChar(8*2+1,8*2+1,93);

	// draw separator

	for (local_x=20;local_x<38;++local_x)
	{
		SetChar(local_x,2,64+2+128);
	}

	print_string(20,0,"Small Reversi",false);
	print_string(20,1,"Jakub Debski '2011",false);
	SetChar(27,2,'{');
}

void system_init()
{
	_graphics(0);
	POKE(COLOR1,0xFF); 
	POKE(COLOR2,0); 
	// Set the font
	POKE(CHBASE_SHADOW,0x20); // font is at 0x2000
	joy_install(&joy_driver);
	video_ptr = (byte * ) SAVMSC;
	draw_game_screen();
	init_first_board();
	find_possible_moves_and_count_pieces();
	copy_new_board_to_visible();
	show_board();
}

byte cursor_x, cursor_y;

void blink_cursor()
{
	local_temp1=GetChar(1+cursor_x*2,1+cursor_y*2);
	local_temp1+=128;
	SetChar(1+cursor_x*2,1+cursor_y*2,local_temp1);
	SetChar(2+cursor_x*2,1+cursor_y*2,local_temp1+1);
	SetChar(1+cursor_x*2,2+cursor_y*2,local_temp1+2);
	SetChar(2+cursor_x*2,2+cursor_y*2,local_temp1+3);
}

bool get_player_action()
{
	int joy;
	cursor_x=last_x;
	cursor_y=last_y;
	blink_cursor();
	POKE(CDTMV3,10);
	for(;;)
	{
		if (PEEK(CDTMV3)==0)
		{
			POKE(CDTMV3,10);
			blink_cursor();
		}

		joy=joy_read(JOY_1);

		// exit when escape pressed
		if (PEEK(KBCODE)==28)
		{
			POKE(KBCODE,0);
			return false;
		}

		if (JOY_BTN_LEFT(joy) || JOY_BTN_RIGHT(joy) || JOY_BTN_DOWN(joy) || JOY_BTN_UP(joy))
		{
			if (GetChar(1+cursor_x*2,1+cursor_y*2)>=128)
				blink_cursor();
		}

		if (JOY_BTN_LEFT(joy))
		{
			if (cursor_x>0)
				--cursor_x;
			blink_cursor();
			wait(1);
		}
		else if (JOY_BTN_RIGHT(joy))
		{
			if (cursor_x<7)
				++cursor_x;
			blink_cursor();
			wait(1);
		}
		else if (JOY_BTN_UP(joy))
		{
			if (cursor_y>0)
				--cursor_y;
			blink_cursor();
			wait(1);
		}
		else if (JOY_BTN_DOWN(joy))
		{
			if (cursor_y<7)
				++cursor_y;
			blink_cursor();
			wait(1);
		}
		else if (JOY_BTN_FIRE(joy))
		{
			if (visible_board[cursor_y][cursor_x]==POSSIBLE_MOVE)
			{
				move_x=cursor_x;
				move_y=cursor_y;
				return true;
			}
		}
	};
	return false;
}

void clear_right_menu()
{
	for (cursor_y=4;cursor_y<=13;++cursor_y)
	{
		memset(video_ptr+40*cursor_y+20,0,20);
	}
}

void winner(byte who)
{
	char *w;
	switch(who)
	{
		case WHITE:
			w="WHITE";
			break;
		case BLACK:
			w="BLACK";
			break;
		default:
			print_string(20,12,"DRAW!",false);
			return;
	}
	print_string(20,12,w,false);
	print_string(26,12,"WINS!",false);
	wait_for_fire();
	clear_right_menu();
}

void mark_last_move()
{
	if (last_x==0xFF)
		return;
	if (last_color==WHITE)
		local_temp1=76;
	else
		local_temp1=80;
	SetChar(1+last_x*2,1+last_y*2,local_temp1);
	SetChar(2+last_x*2,1+last_y*2,local_temp1+1);
	SetChar(1+last_x*2,2+last_y*2,local_temp1+2);
	SetChar(2+last_x*2,2+last_y*2,local_temp1+3);
}

#define MAX_OPTIONS 4

void print_options()
{
	print_string(20,5,"BLACK:",false);
	print_string(27,5,"Player",!black_computer_controlled);
	print_string(34,5,"CPU",black_computer_controlled);
	
	print_string(20,7,"WHITE:",false);
	print_string(27,7,"Player",!white_computer_controlled);
	print_string(34,7,"CPU",white_computer_controlled);

	print_string(20,9,"LEVEL:",false);
	print_string(27,9,"1",tree_depth==1);
	print_string(30,9,"2",tree_depth==2);
	print_string(33,9,"3",tree_depth==3);

	print_string(20,11,"FIRST:",false);
	print_string(27,11,"Black",current_color==BLACK);
	print_string(33,11,"White",current_color==WHITE);

	print_string(20,13,"Start Game",local_temp1==cursor_y);
}

void show_menu_cursor()
{
	SetChar(19,5+cursor_y*2,'|');
	SetChar(38,5+cursor_y*2,'}');
	wait(1);
}

void clear_menu_cursor()
{
	SetChar(19,5+cursor_y*2,0);
	SetChar(38,5+cursor_y*2,0);
}


void configure_game()
{
	int joy;

	current_color=BLACK;

	clear_right_menu();
	cursor_y=0;
	print_options();
	show_menu_cursor();

	for(;;)
	{
		joy=joy_read(JOY_1);
		if (JOY_BTN_LEFT(joy))
		{
			switch(cursor_y)
			{
				case 0:
					black_computer_controlled=false;
					break;
				case 1:
					white_computer_controlled=false;
					break;
				case 2:
					if (tree_depth>1)
						--tree_depth;
					wait(1);
					break;
				case 3:
					current_color=BLACK;
					find_possible_moves_and_count_pieces();
					copy_new_board_to_visible();
					show_board();
					break;
			}
			print_options();
		}
		else if (JOY_BTN_RIGHT(joy))
		{
			switch(cursor_y)
			{
				case 0:
					black_computer_controlled=true;
					break;
				case 1:
					white_computer_controlled=true;
					break;
				case 2:
					if (tree_depth<3)
						++tree_depth;
					wait(1);
					break;
				case 3:
					current_color=WHITE;
					find_possible_moves_and_count_pieces();
					copy_new_board_to_visible();
					show_board();
					break;
			}
			print_options();
		}
		else if (JOY_BTN_UP(joy))
		{
			if (cursor_y!=0)
			{
				clear_menu_cursor();
				--cursor_y;
				show_menu_cursor();
			}
		}
		else if (JOY_BTN_DOWN(joy))
		{
			if (cursor_y<MAX_OPTIONS)
			{
				clear_menu_cursor();
				++cursor_y;
				show_menu_cursor();
			}
		}
		else if (JOY_BTN_FIRE(joy))
		{
			if (cursor_y==MAX_OPTIONS)
				break;
		}
	};
	clear_menu_cursor();
	clear_right_menu();

//	if (tree_depth==3)
//		tree_depth=4;
}
